/** @file   carnivoreai.cpp
 * @brief   Implementation of CarnivoreAI - class.
 * @version $Revision: 1.5 $
 * @author  Tomi Lamminsaari
 */

#include "carnivoreai.h" // class's header file
#include "www_assert.h"
#include "utils.h"
#include "warglobals.h"
#include "gameobject.h"
using std::vector;
using namespace eng2d;

namespace WeWantWar {


/** Constructor.
 */
CarnivoreAI::CarnivoreAI( GameObject* pAlien ) :
  AIController( pAlien, WarGlobals::pObjTable->pPlayer ),
  
  m_alerted( false ),
  m_sprintDistance( 60 )
{
  AIController::m_visualRange = 500 ;
  AIController::m_attackRange = 30;
  WWW_ASSERT( pAlien != 0 );
}



/** Constructor.
 */
CarnivoreAI::CarnivoreAI( GameObject* pAlien, GameObject* pTarget ) :
  AIController( pAlien, pTarget ),
  m_alerted( false ),
  m_sprintDistance( 60 )
{
  AIController::m_visualRange = 500;
  AIController::m_attackRange = 30;
}



/** Destructor.
 */
CarnivoreAI::~CarnivoreAI()
{
}



/** Controls the Carnivore Alien
 */
void CarnivoreAI::updateAI( bool seeTarget, float dist )
{
  // We need to find new controls to the alien. First we set the random
  // movement. If the total distace from this alien to player is
  // big enough, we move randomly.
  this->setRandomMovement( (rand() % 20) + 5 );
  this->avoidWalls();
  
  if ( m_pTarget == 0 ) {
    // No target so we move randomly.
    return;
  }
  
  // If we're stuck we should do something to unstuck us.
  if ( m_stuck == true ) {
    m_alerted = false;
    m_stuck = false;
    return;
  }
  
  // If we can see the player we store the position where we saw him.
  if ( seeTarget == true ) {
    m_lastKnownPos = m_pTarget->position();
    m_alerted = true;
  }
  
  if ( m_alerted == false ) {
    // We've not seen player for a while so we move randomly
    return;
  }
  
  // Are we within the attacking range.
  if ( this->withinAttackRange() == true ) {
    if ( seeTarget == true ) {
      // We're within the attacking range and we can see the player.
      // We attack.
      this->reset();
      this->shoot() = 1;
      this->setCounter( 9 );
      return;
    }
    
    // We're wthin the shooting range but we don't see the player. We call
    // of the alert.
    m_alerted = false;
    return;
  }
  
  // Find the shortest path to the player.
  AStarAlgorithm algo( *this );
  algo.SetMaxRouteNodes( 32 );
  bool routeFound = algo.FindPath( BaseController::m_pObject->position(),
                                   m_pTarget->position() );
  const MPathFinderAlgo::NodeArray& nodes = algo.GetPathNodes();
  if ( routeFound == false || nodes.size() < 2 ) {
    // No route to player. Move randomly.
    m_alerted = false;
    return;
  }
  // The A* algorithm creates the whole path to the target position. We need
  // only the first node.
  Vec2D targetCoord = nodes.at(1);
  this->reset();
  this->forward() = 1;
  int d = Utils::findTurningDir( BaseController::m_pObject->position(),
                                 targetCoord,
                                 BaseController::m_pObject->angle() );
  this->turn() = 3 * d;
  this->setCounter( 3 + rand() % 3 );
}


} // end of namespace

/**
 * Version history
 * ===============
 * $Log: carnivoreai.cpp,v $
 * Revision 1.5  2006/05/15 19:22:53  lamminsa
 * Little tuning.
 *
 * Revision 1.4  2006/04/03 20:57:37  lamminsa
 * Lesser route nodes.
 *
 * Revision 1.3  2006/02/26 07:36:03  lamminsa
 * This class no longer implements the eng2d::MMapDataProvider
 * interface.
 *
 * Revision 1.2  2006/02/24 12:27:58  lamminsa
 * no message
 *
 * Revision 1.1.1.1  2006/01/21 23:02:43  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:15:34+02  lamminsa
 * Initial revision
 *
 */
